<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-project-innovation-initiative
* @subpackage controllers
* @filesource
*//** */

require_once APPPATH.'controllers/adminpanel.php';

/**
* Admin pages for group management.
*
* This code was originally in the AdminPanel controller, but have been moved to a child class for better maintainability / code organization.
* Note that many of these functions were originally in one giant function called manage_groups() which routed calls by action to helper methods; 
* routing is now handled using Codeigniter routing functionality instead so that each action can be separated into a child function.
*
* @todo Many of the forms direct to {@link process_request}, which routes to an appropriate field based on the $_POST function. It would be better
* to set up the forms to point directly to their own processing action and remove the process_request function.
*
* @todo Remove process_request function and set up forms to go directly to the methods that process_request would have routed them to
*
* @package direct-project-innovation-initiative
* @subpackage controllers
*/
class Group_controller extends AdminPanel {
	
	protected $validation_errors = array();
	
	function __construct() {
		parent::__construct();
		$this->verify_permission('manage_groups'); 
	} 


	/**
	* Displays a table of all the groups for this application.
	* By default, shows the un-archived groups; if $show_removed is set to true, shows the archived groups.
	*
	* @param int 
	* @param boolean 
	*/	
	function index($page_number=1, $show_removed = false){
		if(!$this->is->nonzero_unsigned_integer($page_number)) redirect('adminpanel/manage_groups');
		if(!is_bool($show_removed))	$this->error->warning->should_be_a_boolean($show_removed);

		$data['title'] = PORTAL_TITLE_PREFIX . 'Admin Panel'; //set title of page
		
		//pull group information from ldap
		$dn = LDAP_GROUPS_DN;
		if($show_removed){
			$dn = LDAP_DELETED_GROUPS_DN;
		}
		if($this->ldap->connected()) {
			$ldap_result = $this->ldap->search(NULL,NULL,array('cn','ou','description'),'(&(ObjectClass=groupofNames))',$dn);
			$i = 0;
			$groups = $this->group_access($ldap_result,$show_removed);
			$data['groups'] = array_slice($groups,(($page_number-1)*ADMINPANEL_DISPLAY_PER_PAGE),ADMINPANEL_DISPLAY_PER_PAGE);
			$data['group_count'] = count($groups);
			$data['page'] = $page_number;
			//($data['groups'] Sort result table by displayName
			foreach ($data['groups'] as $key => $rowGroups) {
				$displayName[$key]  = $rowGroups['displayname'];
			}
			if($data['group_count']){
				array_multisort($displayName, SORT_ASC, $data['groups']);
			}
		}
		array_walk_recursive($data['groups'], function (&$value) {
			$value = htmlentities($value);
		});
		$this->load->view('adminpanel/manage_groups/'.(($show_removed)?"removed":"index"),$data); //load index view		
	}
	
	
	function create(){
		$this->verify_permission('manage_groups_create');
		//$this->verify_permission('manage_groups_all');
		//note - this method originally didn't let the user select a facility without having the manage_groups_all permission, meaning that the user couldn't create a group without that permission.
		//For the moment, I'm removing that restriction, but we may need to either add it to the verify permission at the top of this method OR remove the requirement that facility be selected
		//and only provide that field when the user has the ability to view facilities.  -- MG 2014/10/17
		
		$this->load->library('audit'); //load audit library
		
		$data['title'] = PORTAL_TITLE_PREFIX . 'Admin Panel'; //set title of page	
		$data['facilities'] = $this->get_facilities();

		//set valdiation rules

		$this->form_validation->set_rules('group_name','Group Name','required|callback_allowed_mailbox_characters|max_length[50]'); //group name can't contain spaces and must be alphanumeric
		$this->form_validation->set_rules('group_display_name','Group Display Name','required|max_length[200]');
		$this->form_validation->set_rules('group_description','Group Description','required|max_length[4000]');
		if(!empty($data['facilities'])) {
			$this->form_validation->set_rules('facility_id','Facility','numeric|required'); //don't require this field if user doesn't have permission to edit it
		}
		
		//if the form is being submitted
		if($this->form_validation->run()) {
			//check if same group already exists in the removed groups dn
			$ldap_result = $this->ldap->search(NULL,NULL,array('dn'),'(&(ObjectClass=groupofNames)(ou=' . $this->input->post('group_name',TRUE) . '))',LDAP_DELETED_GROUPS_DN);
			//only create it if its not already existing, either in the groups dn or deleted groups dn
			if(count($ldap_result) <= 0) { 
				//set attributes for ldap group object
				$attributes['objectclass'] = 'groupOfNames';
				$attributes['member'] = 'cn=admin,' . LDAP_BASE_DOMAIN;
				$attributes['ou'] = $this->input->post('group_name',TRUE);
				$attributes['cn'] = $this->input->post('group_display_name',TRUE);
				$attributes['description'] = $this->input->post('group_description',TRUE);
				//set attributes for ldap group user object (so group information appears in ldap searches for users in global address book
				//and dovecot will deliver mail to the group address)
				$user_attributes["objectClass"] = array('posixAccount', 'top', 'person', 'organizationalPerson', 'inetOrgPerson');
				$user_attributes["gidNumber"] = '5000';
				$user_attributes["uidNumber"] = '5000';
				$user_attributes["uid"] =  $attributes['ou'];
				$user_attributes["homeDirectory"] = '/var/mailboxes/' . strtolower($attributes['ou']);
				$user_attributes["cn"] = $attributes['cn'];
				$user_attributes["displayName"] = $attributes['cn'];
				$user_attributes["givenName"] = $attributes['ou'];
				$user_attributes["sn"] = $attributes['ou'];
				$user_attributes["mail"] = $attributes['ou'] . "@" . DIRECT_DOMAIN;
				//set attributes for database
				$ext_mail = '';
				$services = array_intersect_key($this->input->post(null,TRUE), array_flip(preg_grep('/^service_/', array_keys($this->input->post(null,TRUE)))));
				if(!User::exists(array('user_name' => $attributes['ou']))) {
					//create the group first
					$response = $this->create_group($attributes['ou'], $attributes['cn'], $attributes['description'], $this->input->post('facility_id',TRUE),$services);
					if($response && $response->http_status === 200){
						if($this->ldap->create_group($attributes)) { 
							//then create the user account
							if($this->ldap->create_ldap_account($user_attributes)) { 
								//log event
								if(!$this->audit->log_event('edit',array(0,$this->user->id,'Created group: ' . $attributes['ou'],date('U')))) {
									log_message('error','Failed to audit create group for group: ' . $attributes['ou']);
								}
								$db_attributes = array(
									'username' => $attributes['ou'],
									'ext_mail' => '',
									'pass' => '',
									'actor_id' => $this->user->id,
									'user_ext_notify_flag' => 1,
									'user_ext_group_notify_flag' => 1,
									'user_deleted_flag' => 0,
									'user_is_group' => 1,
								);
								$create_user_rec = $this->webmailmodel->create_user_record($db_attributes);
								if($create_user_rec) {
									$this->session->set_success_message('Created group successfully.');
								}
								else {
									$this->session->set_error_message('Failed to create group user account, contact system administrator.');
								}
							}
							//if user account creation fails
							else { 
								$ldap_error = strtolower($this->ldap->get_ldap_error());
								$reason = '';
								if(strpos($ldap_error,'already exists') !== FALSE) { $reason = ' Group user account already exists.'; }
								$this->session->set_error_message('Failed to create group. '.$reason);
							}
						}
						//if group creation fails
						else {
							$ldap_error = strtolower($this->ldap->get_ldap_error());
							$reason = '';
							if(strpos($ldap_error,'already exists') !== FALSE) { $reason = ' Group user account already exists.'; }
							$this->session->set_error_message('Failed to create group. '.$reason);
						}
					}
					else{
						if($response->http_status === 404){
							$this->session->set_error_message('Failed to create group.  Unable to contact API');
						}
						else{
							$this->session->set_error_message('Failed to create group. '.$response->response->message);
						}
					}
				}
				//if group username is taken by regular user
				else {
					$ldap_error = strtolower($this->ldap->get_ldap_error());
					$reason = '';
					if(strpos($ldap_error,'already exists') !== FALSE) { $reason = ' Group already exists.'; }
					$this->session->set_error_message('Failed to create group.'.$reason);
				}
			}
			else {
				$this->session->set_error_message('Failed to create group, group name already exists in removed groups.');
			}
			
			redirect('adminpanel/manage_groups');
		}

		//if validation fails or form is not being submitted
		$data['form_data'] = $form_data = $this->input->post(NULL,TRUE);
		
		$data['services'] = array();		
		$services =  $this->api_model->webservice_call('/admin/web_services/format/json', 'GET');
		if($services->http_status === 200){
			$data['services'] = $services->response->services;
		}
		
		$data['validation_errors'] = array();
		foreach(array_keys($_POST) as $field){
			if(!empty(form_error($field))) { $data['validation_errors'][$field] = form_error($field); }
		}
		
		$this->load->view('adminpanel/manage_groups/create',$data); //load view
	}
	
	function edit($value = NULL, $subvalue = NULL){
		if(is_null($value)) show_404();
		if(!empty($_POST) && empty($this->validation_errors)) return $this->update(($value == 'removed'));
		

		$data['title'] = PORTAL_TITLE_PREFIX . 'Admin Panel'; //set title of page
		$data['is_active'] = ($value != 'removed');
		
		if($value == 'removed') {
			$dn = LDAP_DELETED_GROUPS_DN;
			if(isset($subvalue)) { $group_name = $subvalue; } else { show_404(); }
			$this->can_edit_group($group_name, true);
		}
		else {
			$dn = LDAP_GROUPS_DN;
			$group_name = $value;
			$this->can_edit_group($group_name, false);
		}
		
		//get group information from ldap
		$response = $this->api_model->webservice_call('/direct/group/mailbox/'.$group_name.'/format/json','GET');
		if($response->http_status === 200){
			$data['management'] = true;
			$data = array_merge($data,get_object_vars($response->response));
		}
		else{
			$data['management'] = false;
		}
		$data['facilities']=$this->get_facilities();
		
		if($this->ldap->connected()) {
			$ldap_result = $this->ldap->search(NULL,NULL,array('ou','member'),'(&(ObjectClass=groupofNames)(ou=' . $group_name . '))',$dn);
			if(count($ldap_result) > 0) { 
				$data['members'] = array();
				//get group member info from ldap
				if(is_array($ldap_result[0]['member'])) {
					foreach($ldap_result[0]['member'] as $member) {
						$username = str_replace('uid=','',$member);
						$username = str_replace(','.LDAP_ACCOUNTS_DN,'',$username);
						//check that user value corresponds to valid user
						$get_user = $this->db->query("SELECT user_name FROM users WHERE user_deleted_flag=0 AND user_name=" . $this->db->escape($username));
						if($get_user) {
							if($get_user->num_rows() == 1) {
								$user_result = $this->ldap->search(null,1,array('displayname','uid'),'(uid='.$username.')');
								if(count($user_result) > 0) {
									$member = array('displayname' => $user_result[0]['displayname'], 'uid' => $user_result[0]['uid']);
								}
								else { $member = array('displayname' => '', 'uid' => $username); }
								array_push($data['members'],$member);
							}
						}
					}
				}
				$get_ext_email = $this->db->query('SELECT user_ext_notify_flag FROM users WHERE user_name='.$this->db->escape($group_name).' AND user_is_group=1');
				if($get_ext_email) {
					if($get_ext_email->num_rows() == 1) {
						$row = $get_ext_email->row_array();
						$data['group_notify'] = $row['user_ext_notify_flag'];
					}
				}
			}
			else { show_404(); /*if not actual group, show 404*/ }
		}
		
		$data['group_name'] = $group_name;
		$data['validation_errors'] = $this->validation_errors;
		$data['form_values'] = array();
		
		foreach(array('display_name', 'description', 'facility_id', 'group_notify') as $field){
			if(!empty($_POST))
				$data['form_values'][$field] = element($field, $_POST); //don't cleanse - we want the raw value when redisplaying user input fields
			else
				$data['form_values'][$field] = $data[$field];
		}
				
		
		$this->load->view('adminpanel/manage_groups/edit_group',$data); //load view	
	}
	
	function group($value = NULL, $subvalue = NULL){
		$this->edit($value, $subvalue);
	}	

	
	function removed($page_number=1){
		if(!$this->is->nonzero_unsigned_integer($page_number)) redirect('adminpanel/manage_groups/removed');
		$this->index($page_number, $show_removed=true);
	}

	
/////////////////////////////////////////////////
// PROTECTED METHODS
// Helper methods for this class which are not web-accessible
/////////////////////////////////////////////////	

	
	protected function add_user(){
		$data['title'] = PORTAL_TITLE_PREFIX . 'Admin Panel'; //set title of page
		
		$group = $this->input->post('group',TRUE);
		$users = $this->input->post('users',TRUE);
		$active = $this->input->post('active',TRUE);
		$this->can_edit_group($group, $active == 'FALSE');
		//check whether we are acting on multiple users (multi-select input or not)
		if(is_array($users)) {
			foreach($users as $user) {
				//add member to group, then log the change if successful
				if($active == 'FALSE') { $dn = LDAP_DELETED_GROUPS_DN; } else { $dn = NULL; }
				if($this->ldap->add_group_membership($group,$user,$dn) !== FALSE) {
					if(!$this->audit->log_event('edit',array($this->get_user_id($user),$this->user->id,'Add user to group: ' . $group,date('U')))) {
						log_message('error','Failed to audit add user event for user: ' . $user . ' to group: ' . $group);
					}
				}
				//if unsuccessful, see if its due to already existing (other common reasons can be checked for here too) and set error message
				else {
					$ldap_error = strtolower($this->ldap->get_ldap_error());
					$reason = '';
					if(strpos($ldap_error,'exists') !== FALSE) { $reason = ' User is already in group.'; }
					$this->session->set_flashdata('message','Failed to add user: '.$user.' to group.'.$reason);
					$this->session->set_flashdata('message_class','error'); 
				}
			}
		}
		else { 
			//add member to group, then log the change if successful
			if($active == 'FALSE') { $dn = LDAP_DELETED_GROUPS_DN; } else { $dn = NULL; }
			if($this->ldap->add_group_membership($group,$users,$dn) !== FALSE) {
				if(!$this->audit->log_event('edit',array($this->get_user_id($users),$this->user->id,'Add user to group: ' . $group,date('U')))) {
					log_message('error','Failed to audit add user event for user: ' . $users . ' to group: ' . $group);
				}
			}
			//if unsuccessful, see if its due to already existing (other common reasons can be checked for here too) and set error message
			else {
				$ldap_error = strtolower($this->ldap->get_ldap_error());
				$reason = '';
				if(strpos($ldap_error,'exists') !== FALSE) { $reason = ' User is already in group.'; }
				$this->session->set_flashdata('message','Failed to add user: '.$user.' to group.'.$reason);
				$this->session->set_flashdata('message_class','error'); 
			}
		}
		if($active == 'TRUE') { redirect('adminpanel/manage_groups/group/' . $group); }
		else { 
			redirect('adminpanel/manage_groups/group/removed/' . $group); 
		}
	}
	
	protected function can_edit_group($group_name, $removed){
		if($this->has_permission("manage_groups_all")){
			return true;
		}
		if($this->has_permission("manage_groups_facilities")){
			
			$resource = '/direct/group/in_facility/mailbox/'.$this->session->userdata('username');
			$resource .= '/format/json';
			$response = @$this->api_model->webservice_call($resource,'GET');
			if($response->http_status === 200){
				$groups = $response->response->accounts;
				if( in_array($group_name, $groups)){
					return true;
				}
			}
		}
		if($this->has_permission("manage_groups_groups")){
			
			if($this->user_in_group($group_name, $removed)){
						return true;
			}
		}
		return show_404();
	}
	
	protected function create_group($group_name,$display_name,$description, $facility,$services) {
		$this->verify_permission("manage_groups_create");
		if(!$this->has_permission("manage_groups_all")){
			$resource = '/direct/account/mailbox/'.$this->session->userdata('username');
			$resource .= '/format/json';
			$response = @$this->api_model->webservice_call($resource,'GET');
			if($response->http_status === 200){
				$facility = $response->response->facility_id;	
			}else{
				return false;
			}
		}
		$mailbox_group = $this->session->userdata('mailbox_group');
		$resource = '/admin/create_group/format/json';
	
		//construct post request
		$fields = array(
			'group_name' => $group_name,
			'display_name' => $display_name,
			'description' => $description,
			'facility' => $facility,
		);
		foreach ($services as $key=>$value){
			$fields[$value]= true;
		}
		$post = '';
		foreach($fields as $key => $value) { $post .= $key.'='.urlencode($value).'&'; } //url-ify the data for the POST
		$post = rtrim($post, '&');
		return $this->api_model->webservice_call($resource,'POST',$post);
	}		
	
	protected function get_facilities(){
		$facilities = array();
		$api_result = $this->api_model->webservice_call('/admin/facilities/format/json','GET');
		if($api_result->http_status === 200){
			foreach($api_result->response->facilities as $facility){
				$facilities[$facility->id] = $facility->name;
			}
		}
		return $facilities;
	}

	//remove group from system (or archive?  are we removing or archiving?)
	protected function remove(){			
		$data['title'] = PORTAL_TITLE_PREFIX . 'Admin Panel'; //set title of page	
	
		$group = $this->input->post('group',TRUE);
		$this->can_edit_group($group, false);
		if($this->ldap->delete_ldap_account($group) !== FALSE && $this->ldap->remove_group($group) !== FALSE) {
			if(!$this->audit->log_event('edit',array(0,$this->user->id,'Remove group: ' . $group,date('U')))) {
				log_message('error','Failed to audit remove group event for group: ' . $group);
			}
			if($this->session->userdata('mailbox_group') == $group) {
				$this->session->set_mailbox_location('inbox');;
				$this->session->set_userdata('mailbox_group',$this->session->userdata('username'));
				$this->session->set_userdata('mailbox_group_cn',$this->session->userdata('username'));
			}
			$this->session->set_success_message('Group successfully removed');
			redirect('adminpanel/manage_groups/');
		}
		else {
			$this->session->set_error_message('Failed to remove group. '.$this->ldap->get_ldap_error());
			redirect('adminpanel/manage_groups/group/'.$group);
		}
	}
	
	protected function group_access($ldap_result, $removed){
		$permissiosn = false;
		$groups = array();
		$facility_groups = array();
		if($this->has_permission("manage_groups_all")){
			$permissiosn = true;
		}
		else if($this->has_permission("manage_groups_facilities")){
			$resource = '/direct/group/in_facility/mailbox/'.$this->session->userdata('username');
			$resource .= '/format/json';
			$response = @$this->api_model->webservice_call($resource,'GET');
			if($response->http_status === 200){
				$facility_groups = $response->response->accounts;
			}
		}
		$i=0;
		foreach($ldap_result as $result) {
			if($permissiosn || in_array($result['ou'],$facility_groups) || ($this->has_permission("manage_groups_groups")&&$this->user_in_group($result['ou'],$removed))){
				$groups[$i]['name'] = $result['ou'];
				$groups[$i]['displayname'] = $result['cn'];
				$groups[$i]['description'] = isset($result['description']) ? $result['description'] : '';
				$i++;
			}
		}
		return $groups;
	}	
	
	//remove user from group
	protected function remove_user(){
		$data['title'] = PORTAL_TITLE_PREFIX . 'Admin Panel'; //set title of page
		
		$group = $this->input->post('group',TRUE);
		$users = $this->input->post('group_users',TRUE);
		$active = $this->input->post('active',TRUE);
		$this->can_edit_group($group, $active == 'FALSE');
		//check whether we are acting on multiple users (multi-select input or not)
		if(is_array($users)) {
			if($active == 'FALSE') { $dn = LDAP_DELETED_GROUPS_DN; } else { $dn = NULL; }
			foreach($users as $user) {
				if($this->ldap->remove_group_membership($group,$user,$dn)) {
					if(!$this->audit->log_event('edit',array($this->get_user_id($user),$this->user->id,'Remove user from group: ' . $group,date('U')))) {
						log_message('error','Failed to audit remove user event for user: ' . $user . ' from group: ' . $group);
					}
					if($user == $this->session->userdata('username')) {
						if($this->session->userdata('mailbox_group') == $group) {
							$this->session->set_mailbox_location('inbox');
							$this->session->set_userdata('mailbox_group',$this->session->userdata('username'));
							$this->session->set_userdata('mailbox_group_cn',$this->session->userdata('username'));
						}
					}
				}
				else { 
					$this->session->set_flashdata('message','Failed to remove user: '.$user.' from group.');
					$this->session->set_flashdata('message_class','error');  
				}
			}
		}
		else { 
			if($active == 'FALSE') { $dn = LDAP_DELETED_GROUPS_DN; } else { $dn = NULL; }
			if($this->ldap->remove_group_membership($group,$users,$dn)) {
				if(!$this->audit->log_event('edit',array($this->get_user_id($users),$this->user->id,'Remove user from group: ' . $group,date('U')))) {
						log_message('error','Failed to audit remove user event for user: ' . $users . ' from group: ' . $group);
				}
				if($users == $this->session->userdata('username')) {
					if($this->session->userdata('mailbox_group') == $group) {
						$this->session->set_mailbox_location('inbox');;
						$this->session->set_userdata('mailbox_group',$this->session->userdata('username'));
						$this->session->set_userdata('mailbox_group_cn',$this->session->userdata('username'));
					}
				}
			}
			else { 
				$this->session->set_flashdata('message','Failed to remove user: '.$users.' from group.');
				$this->session->set_flashdata('message_class','error');  
			}
		}
		if($active == 'TRUE') { redirect('adminpanel/manage_groups/group/' . $group); }
		else { redirect('adminpanel/manage_groups/group/removed/' . $group); }
	}
	
	
	protected function restore(){
		$data['title'] = PORTAL_TITLE_PREFIX . 'Admin Panel'; //set title of page
		
		$group = $this->input->post('group',TRUE);
		$this->can_edit_group($group, true);
		if($this->ldap->restore_group($group) !== FALSE && $this->ldap->restore_ldap_account($group) !== FALSE) {
			if(!$this->audit->log_event('edit',array(0,$this->user->id,'Restore group: ' . $group,date('U')))) {
				log_message('error','Failed to audit restore group event for group: ' . $group);
			}
			redirect('adminpanel/manage_groups/');
		}
		else {
			$this->session->set_error_message('Failed to restore group. '.$this->ldap->get_ldap_error());
			redirect('adminpanel/manage_groups/removed/'.$group);
		}	
	}			
	
	protected function update($removed=false){
		if(empty($_POST)) show_404();		
				
		$this->load->library('audit'); //load audit library for tracking changes to groups
		
		if(array_key_exists('add_user',$_POST)) return $this->add_user();
		if(array_key_exists('remove_user',$_POST)) return $this->remove_user();
		if(array_key_exists('remove',$_POST)) return $this->remove(); //archive the group
		if(array_key_exists('restore',$_POST)) return $this->restore();
		if(array_key_exists('update_removed',$_POST)) $removed = true; 
		
		$data['title'] = PORTAL_TITLE_PREFIX . 'Admin Panel'; //set title of page		
		
		$group_name = $this->input->post('group',TRUE);
		$management = ($this->input->post('management',TRUE) === "TRUE");
		$group_display_name = $this->input->post('display_name',TRUE);
		$group_description = $this->input->post('description',TRUE);
		if(array_key_exists('group_notify',$this->input->post(NULL,TRUE))) { $group_notify = TRUE; } else { $group_notify = FALSE; }
		$attributes = array('cn' => $group_display_name, 'description' => $group_description); //set ldap attributes
		
		//set valdiation rules
		$this->form_validation->set_rules('display_name','Display Name','required');
		$this->form_validation->set_rules('description','Description','required');
		$this->form_validation->set_rules('facility_id','Facility','required');
		
		if(!$this->form_validation->run()){
			//validation fails
			$form_data = $this->input->post(NULL,TRUE);
			$validation_errors = array();
			if(is_array($form_data)) {
				foreach($form_data as $key => $input) {
					if(!empty(form_error($key))) { $validation_errors[$key] = form_error($key); }
				}
				$this->validation_errors = $validation_errors;
			}
			
			if($removed)
				return $this->edit('removed', $group_name);
			else
				return $this->edit($group_name);
		}
		
		$fields = $this->input->post();
		//construct post request
		$fields ['mailbox'] = $group_name;
		$post = '';
		foreach($fields as $key => $value) { $post .= $key.'='.urlencode($value).'&'; } //url-ify the data for the POST
		$post = rtrim($post, '&');
		if($management){
			$response = $this->api_model->webservice_call('/direct/group/update/format/json','POST',$post);
		}
		if(!$management || $response->http_status === 200){
			$update_db = $this->db->query('UPDATE users SET user_ext_notify_flag='.$this->db->escape($group_notify).' WHERE user_name='.$this->db->escape($group_name));
			if($update_db) { //if db updates, update ldap
				if($removed){
					$success = $this->ldap->modify_group($group_name,$attributes,LDAP_DELETED_GROUPS_DN);
				}
				else{
					$success = $this->ldap->modify_group($group_name,$attributes);
				}
				if($success) { 
					$this->session->set_success_message('Updated group information successfully.');
					if(!$this->audit->log_event('edit',array(0,$this->user->id,'Updated group information: ' . $group_name,date('U')))) {
						log_message('error','Failed to audit update group information for group: ' . $group_name);
					}
				}
				else { 
					$this->session->set_error_message('Failed to update group.');
				}
			}
			else { 
				$this->session->set_error_message('Failed to update group.');;
				
			}
		}
		else {
			$this->session->set_error_message('Failed to update group.');
				
		}
		redirect('adminpanel/manage_groups/group/'.($removed?"removed/":"") . $group_name);
	}		
	
	protected function user_in_group($group_name , $removed){
		$dn = LDAP_GROUPS_DN;
		if($removed)$dn = LDAP_DELETED_GROUPS_DN;
		$ldap_result = $this->ldap->search(NULL,NULL,array('ou','member'),'(&(ObjectClass=groupofNames)(ou=' . $group_name . '))',$dn);
		if(is_array($ldap_result[0]['member'])) {
			foreach($ldap_result[0]['member'] as $member) {
				$username = str_replace('uid=','',$member);
				$username = str_replace(','.LDAP_ACCOUNTS_DN,'',$username);
				if($this->session->userdata('username') === $username){
					return true;
				}
			}
		}
		return false;
	}	
	
}
/* End of file adminpanel.php */
/* Location: ./application/controllers/adminpanel.php */